from flask import Flask, Response, request
import subprocess, shlex, os, time, unicodedata, glob, threading
from urllib.parse import unquote
import yt_dlp

app = Flask(__name__)
MUSIC_DIR = "./music"
os.makedirs(MUSIC_DIR, exist_ok=True)

# ===== Simple in-process download locks =====
DOWNLOAD_LOCKS = {}
LOCKS_GUARD = threading.Lock()

def get_lock(key: str) -> threading.Lock:
    with LOCKS_GUARD:
        if key not in DOWNLOAD_LOCKS:
            DOWNLOAD_LOCKS[key] = threading.Lock()
        return DOWNLOAD_LOCKS[key]

# ===== Normalize / fuzzy helpers =====
def strip_accents(text: str) -> str:
    text = unicodedata.normalize("NFD", text)
    return text.encode("ascii", "ignore").decode("utf-8")

def normalize_key(text: str) -> str:
    t = strip_accents(text).lower()
    for ch in ["-", "_"]:
        t = t.replace(ch, " ")
    t = " ".join(t.split())  # collapse spaces
    return t

def filename_for(key: str) -> str:
    # lưu dạng: từ-khóa-không-dấu, nối bằng underscore cho nhất quán
    return normalize_key(key).replace(" ", "_") + ".mp3"

def find_existing_file(query: str) -> str | None:
    """
    Tìm file đã có theo fuzzy:
    - Ưu tiên trùng toàn bộ key chuẩn.
    - Chấp nhận khớp >= 60% token, bỏ qua khớp 1 từ duy nhất.
    """
    if not os.path.isdir(MUSIC_DIR):
        return None

    norm_query = normalize_key(query)
    tokens = norm_query.split()

    # 1️⃣ Ưu tiên khớp tên chuẩn tuyệt đối
    target = os.path.join(MUSIC_DIR, filename_for(query))
    if os.path.exists(target):
        return target

    best_match = None
    best_score = 0

    # 2️⃣ So sánh fuzzy với từng file
    for f in os.listdir(MUSIC_DIR):
        if not f.lower().endswith(".mp3"):
            continue
        base = os.path.splitext(f)[0]
        norm_base = normalize_key(base)
        base_tokens = norm_base.split()

        # Đếm số token trùng
        common = sum(1 for t in tokens if t in base_tokens)
        if not common:
            continue

        # Tính độ khớp (%)
        score = (common / max(len(tokens), len(base_tokens))) * 100

        # Bỏ qua khớp quá ít (chỉ 1 token đơn)
        if common == 1 and len(tokens) > 2:
            continue

        # Cập nhật nếu tốt hơn
        if score > best_score:
            best_score = score
            best_match = os.path.join(MUSIC_DIR, f)

    # Chỉ nhận nếu score >= 60%
    if best_match and best_score >= 60:
        print(f"🔎 Fuzzy match: {os.path.basename(best_match)} ({best_score:.0f}%)")
        return best_match

    return None

def wait_file_ready(path: str, timeout_sec=20) -> bool:
    base = os.path.splitext(path)[0]
    for _ in range(int(timeout_sec * 2)):  # mỗi 0.5s
        part_exists = glob.glob(base + ".*part*")
        if os.path.exists(path) and not part_exists and os.path.getsize(path) > 1024:
            return True
        time.sleep(0.5)
    return False

# ===== Download via yt_dlp (with Android client to avoid SABR) =====
def download_song(keyword: str) -> str | None:
    """Tải bài hát từ YouTube"""
    safe_name = filename_for(keyword).replace(".mp3", "")  # 👈 bỏ .mp3 ở đây
    output_base = os.path.join(MUSIC_DIR, safe_name)       # không có .mp3
    final_path = output_base + ".mp3"

    if os.path.exists(final_path):
        print(f"✅ Đã có sẵn: {final_path}")
        return final_path

    print(f"🔎 Đang tìm và tải bài: {keyword} ...")
    ydl_opts = {
        "format": "bestaudio/best",
        "outtmpl": output_base,      # 👈 không có .mp3 ở đây
        "quiet": True,
        "noplaylist": True,
        "default_search": "ytsearch1",
        "extractor_args": {"youtube": {"player_client": ["android"]}},
        "postprocessors": [{
            "key": "FFmpegExtractAudio",
            "preferredcodec": "mp3",
            "preferredquality": "128",
        }],
        "merge_output_format": "mp3",
    }

    try:
        with yt_dlp.YoutubeDL(ydl_opts) as ydl:
            try:
                ydl.download([f"{keyword} Zing MP3"])
            except Exception:
                ydl.download([keyword])

        # yt_dlp sẽ tự tạo file .mp3 duy nhất
        possible = [final_path, output_base + ".mp3"]
        for p in possible:
            if os.path.exists(p):
                print(f"🎶 Đã tải xong: {p}")
                if wait_file_ready(p):
                    return p
        print("⚠️ File chưa sẵn sàng sau khi tải.")
        return None
    except Exception as e:
        print(f"❌ Lỗi tải nhạc: {e}")
        return None

def convert_and_stream(file_path: str) -> Response:
    cmd = (
        f'ffmpeg -hide_banner -loglevel error -nostdin -i "{file_path}" '
        f'-vn -acodec pcm_s16le -f wav -ac 1 -ar 24000 '
        f'-af aresample=precision=28 '
        f'-map_metadata -1 -fflags +bitexact -flags +bitexact pipe:1'
    )
    proc = subprocess.Popen(shlex.split(cmd), stdout=subprocess.PIPE, bufsize=4096)

    def generate():
        try:
            while True:
                chunk = proc.stdout.read(4096)
                if not chunk:
                    break
                yield chunk
        finally:
            proc.kill()

    return Response(generate(), mimetype="audio/wav")

# ===== Routes =====
@app.route("/music_wav/<path:name>.wav")
def music_wav(name):
    """
    - Nhận mọi kiểu tên: có dấu/không dấu, '-', '_', khoảng trắng
    - Nếu đã có file phù hợp -> phát ngay
    - Nếu chưa có -> lock theo tên chuẩn -> tải một lần -> phát
    """
    # Chuẩn hóa tên người dùng gõ
    query = unquote(name).strip()
    # Chấp nhận mọi phân tách
    q_print = query
    # Tìm file hiện có (fuzzy)
    existing = find_existing_file(query)

    # Dùng lock theo key chuẩn để tránh tải trùng
    lock_key = normalize_key(query)
    lock = get_lock(lock_key)

    if existing and os.path.exists(existing):
        print(f"🎵 Phát bài (đã có): {os.path.basename(existing)}")
        return convert_and_stream(existing)

    with lock:
        # Double-check sau khi vào lock (tránh race)
        existing = find_existing_file(query)
        if existing and os.path.exists(existing):
            print(f"🎵 Phát bài (đã có, sau lock): {os.path.basename(existing)}")
            return convert_and_stream(existing)

        # Chưa có -> tải
        path = download_song(query)
        if not path or not os.path.exists(path):
            return f"❌ Không tìm thấy hoặc tải thất bại: {q_print}", 404

        print(f"🎵 Phát bài (vừa tải): {os.path.basename(path)}")
        return convert_and_stream(path)

@app.route("/play")
def play_compat():
    f = request.args.get("file", "")
    if not f:
        return "Thiếu tên file", 400
    name = f.replace(".mp3", "").replace(".wav", "")
    return music_wav(name)

if __name__ == "__main__":
    print("🚀 WAV Server ready at http://0.0.0.0:5000")
    app.run(host="0.0.0.0", port=5000, threaded=True)
